<?php
/**
 * @package Mercury Payment Integration
 */

/**
 *
 */
abstract class MercuryRequest
{

    /**
     * Success.
     *
     * @var int
     */
    const RESPONSE_CODE_SUCCESS = 0;

    /**
     * AuthFail (bad password or MerchantId).
     *
     * @var int
     */
    const RESPONSE_CODE_AUTH_FAIL = 100;

    /**
     * CardDeclined – the card was declined for the transaction. Status=Decline.
     *
     * @var int
     */
    const RESPONSE_CODE_CARD_DECLINED = 101;

    /**
     * Mercury Internal Error.
     *
     * @var int
     */
    const RESPONSE_CODE_MERCURY_INTERNAL_ERROR = 200;

    /**
     * FormPostFail – post items invalid/missing.
     *
     * @var int
     */
    const RESPONSE_CODE_FORM_POST_FAIL = 201;

    /**
     * GetInitDBFail – data saved from Initialize Payment invalid/missing.
     *
     * @var int
     */
    const RESPONSE_CODE_GET_INIT_DB_FAIL = 202;

    /**
     * ProcessPaymentFail – unable to process. Payment Status=Error.
     *
     * @var int
     */
    const RESPONSE_CODE_PROCESS_PAYMENT_FAIL = 203;

    /**
     * SavePreAuthDBFail – internal database error for PreAuth transaction.
     *
     * @var int
     */
    const RESPONSE_CODE_SAVE_PRE_AUTH_DB_FAIL = 204;

    /**
     * SaveSaleDBFail – internal database error for Sale transaction.
     *
     * @var int
     */
    const RESPONSE_CODE_SAVE_SALE_DB_FAIL = 205;

    /**
     * ValidationFail: General Validation Error (See Message for list of validation errors).
     *
     * @var int
     */
    const RESPONSE_CODE_VALIDATION_FAIL = 300;

    /**
     * ValidationCCFail – Credit Card failed Mod10 check multiple times.
     *
     * @var int
     */
    const RESPONSE_CODE_VALIDATION_CC_FAIL = 301;

    /**
     * ValidationServerSideFailure – can occur when scripting is off in user’s
     * browser so client side validation did not occur.
     *
     * @var int
     */
    const RESPONSE_CODE_VALIDATION_SERVER_SIDE_FAILURE = 302;

    /**
     * @var \MercuryRequestSpecification|null
     */
     protected $_specification;

    /**
     * @var array|null
     */
    protected $_values;

    /**
     * @var \stdClass|null
     */
    protected $_response;

    /**
     * @param \MercuryRequestSpecification|null [$specification = null]
     * @param array|null [$values = null]
     */
    public function __construct($specification = null, $values = null)
    {
        if (isset($specification)) {
            $this->setSpecification($specification);
        }
        if (isset($values)) {
            $this->setValues($values);
        }
    }

    /**
     * @return array
     */
    public static function getErrorMessages()
    {
        return array(
            self::RESPONSE_CODE_SUCCESS => 'Success',
            self::RESPONSE_CODE_AUTH_FAIL => 'AuthFail (bad password or id)',
            self::RESPONSE_CODE_CARD_DECLINED => 'CardDeclined – the card was declined for the transaction. Status=Decline',
            self::RESPONSE_CODE_MERCURY_INTERNAL_ERROR => 'Mercury Internal Error',
            self::RESPONSE_CODE_FORM_POST_FAIL => 'FormPostFail – post items invalid/missing',
            self::RESPONSE_CODE_GET_INIT_DB_FAIL => 'GetInitDBFail – data saved from Initialize Payment invalid/missing',
            self::RESPONSE_CODE_PROCESS_PAYMENT_FAIL => 'ProcessPaymentFail – unable to process. Payment Status=Error.',
            self::RESPONSE_CODE_SAVE_PRE_AUTH_DB_FAIL => 'SavePreAuthDBFail– internal database error for PreAuth transaction',
            self::RESPONSE_CODE_SAVE_SALE_DB_FAIL => 'SaveSaleDBFail – internal database error for Sale transaction',
            self::RESPONSE_CODE_VALIDATION_FAIL => 'ValidationFail: General Validation Error (See Message for list of validation errors)',
            self::RESPONSE_CODE_VALIDATION_CC_FAIL => 'ValidationCCFail – Credit Card failed Mod10 check multiple times',
            self::RESPONSE_CODE_VALIDATION_SERVER_SIDE_FAILURE => 'ValidationServerSideFailure – can occur when scripting is off in users browser so client side validation did not occur.',
        );
    }

    /**
     * @throws \Exception
     * @return bool
     */
    public function validate()
    {
        if (isset($this->_specification, $this->_values)
            && is_array($this->_values)
            && sizeof($this->_values) > 0
        ) {
            return $this->_specification->validate($this->_values);
        } else {
            Throw new \Exception(
                'No parameters or no values specified in ' . __METHOD__
                . ' in ' . __FILE__ . ' "' . print_r($this, true));
        }
    }

    /**
     * This method should go through values and process them according to specification.
     */
    public function process()
    {
        if (isset($this->_specification)) {
            $this->_values = $this->_specification->process($this->_values);
        }
    }

    /**
     * Each request handles success differently.
     */
    abstract protected function _requestSuccessProcess();

    /**
     * Each request handles failures differently.
     */
    abstract protected function _requestFailureProcess();

    /**
     * @return \stdClass|null
     */
    abstract protected function _request();

    /**
     *
     */
    public function request()
    {
        $this->_response = $this->_request();
        if ($this->responseWasSuccessful()) {
            $this->_requestSuccessProcess();
        } else {
            $this->_requestFailureProcess();
        }
        return $this->responseWasSuccessful();
    }

    /**
     * @param array $values
     */
    public function setValues($values)
    {
        if (isset($values)
            && is_array($values)
        ) {
            $this->_values = $values;
        }
    }

    /**
     * @return array|null
     */
    public function getValues()
    {
        return $this->_values;
    }

    /**
     * @param \MercuryRequestSpecification $specification
     */
    public function setSpecification($specification)
    {
        if (isset($specification)
            && is_a($specification, 'MercuryRequestSpecification')
        ) {
            $this->_specification = $specification;
        }
    }

    /**
     * @return \MercuryRequestSpecification|null
     */
    public function getSpecification()
    {
        return $this->_specification;
    }

    /**
     * @return bool
     */
    public function responseWasSuccessful()
    {
        if (isset($this->_response->ResponseCode)
            && $this->_response->ResponseCode == self::RESPONSE_CODE_SUCCESS
        ) {
            return true;
        }
        return false;
    }

    /**
     * @return null|stdClass
     */
    public function getResponse()
    {
        return $this->_response;
    }

}
