<?php
/**
 * @package Mercury Payment Integration
 */

/**
 *
 */
class InitializePaymentRequest extends MercuryRequest
{

    /**
     * @var string
     */
    const TRAN_TYPE_PRE_AUTH = 'PreAuth';

    /**
     * @var string
     */
    const TRAN_TYPE_SALE = 'Sale';

    /**
     * @var string
     */
    const TRAN_TYPE_RETURN = 'Return';

    /**
     * @var string
     */
    const TRAN_TYPE_VOICE_AUTH = 'VoiceAuth';

    /**
     * @var string
     */
    const FREQUENCY_ONE_TIME = 'OneTime';

    /**
     * @var string
     */
    const FREQUENCY_RECURRING = 'Recurring';

    /**
     * @var string
     */
    const DEFAULT_SWIPE_SWIPE = 'Swipe';

    /**
     * @var string
     */
    const DEFAULT_SWIPE_MANUAL = 'Manual';

    /**
     * @param array|null [$values = null]
     */
    public function __construct($values = null)
    {

        $parameters = array(
            new \MercuryRequestSpecificationParameter(
                'MerchantID',
                \MercuryRequestSpecificationParameter::TYPE_NUMERIC,
                24,
                true
            ),
            new \MercuryRequestSpecificationParameter(
                'Password',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                16,
                true
            ),
            new \MercuryRequestSpecificationParameter(
                'TranType',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                16,
                true
            ),
            new \MercuryRequestSpecificationParameter(
                'TotalAmount',
                \MercuryRequestSpecificationParameter::TYPE_DOUBLE,
                8,
                true
            ),
            new \MercuryRequestSpecificationParameter(
                'PartialAuth',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                3,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'Frequency',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                10,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'VoiceAuthCode',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                10,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'OperatorID',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                16,
                true
            ),
            new \MercuryRequestSpecificationParameter(
                'TerminalName',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                20,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'Invoice',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                16,
                true
            ),
            new \MercuryRequestSpecificationParameter(
                'Memo',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                40,
                true
            ),
            new \MercuryRequestSpecificationParameter(
                'TaxAmount',
                \MercuryRequestSpecificationParameter::TYPE_DOUBLE,
                8,
                true
            ),
            new \MercuryRequestSpecificationParameter(
                'AVSFields',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                4,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'AVSAddress',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                8,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'AVSZip',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                9,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'CustomerCode',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                16,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'Keypad',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                3,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'DefaultSwipe',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                6,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'CardEntryMethod',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                6,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'ProcessCompleteUrl',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                512,
                true
            ),
            new \MercuryRequestSpecificationParameter(
                'ReturnUrl',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                512,
                true
            ),
            new \MercuryRequestSpecificationParameter(
                'PageTimeoutDuration',
                \MercuryRequestSpecificationParameter::TYPE_NUMERIC,
                2,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'PageTimeoutIndicator',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                3,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'DisplayStyle',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                8,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'BackgroundColor',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                16,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'FontColor',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                16,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'FontFamily',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                64,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'FontSize',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                8,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'LogoUrl',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                512,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'PageTitle',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                64,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'OrderTotal',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                3,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'TotalAmountBackgroundColor',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                8,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'SubmitButtonText',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                32,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'CancelButtonText',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                32,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'ButtonTextColor',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                16,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'ButtonBackgroundColor',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                16,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'CancelButton',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                16,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'JCB',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                3,
                false
            ),
            new \MercuryRequestSpecificationParameter(
                'Diners',
                \MercuryRequestSpecificationParameter::TYPE_STRING,
                3,
                false
            ),
        );

        $specification =
            new \MercuryRequestSpecification($parameters);

        parent::__construct($specification, $values);

    }

    /**
     * @return bool
     */
    public function validate()
    {
        if (parent::validate()) {

            $specification = & $this->getSpecification();

            if ($this->_values['TranType'] != self::TRAN_TYPE_PRE_AUTH
                && $this->_values['TranType'] != self::TRAN_TYPE_RETURN
                && $this->_values['TranType'] != self::TRAN_TYPE_SALE
                && $this->_values['TranType'] != self::TRAN_TYPE_VOICE_AUTH
            ) {
                $specification->addValidationError(
                    'TranType', 'Should be ('
                    . self::TRAN_TYPE_PRE_AUTH . ',' . self::TRAN_TYPE_RETURN
                    . ',' . self::TRAN_TYPE_SALE . ' or' . self::TRAN_TYPE_VOICE_AUTH . ')');
                return false;
            }

            if (isset($this->_values['PartialAuth'])
                && ($this->_values['PartialAuth'] != 'on'
                && $this->_values['PartialAuth'] != 'off')
            ) {
                $specification->addValidationError(
                    'PartialAuth', 'Should be on, off');
                return false;
            }

            if ($this->_values['Frequency'] != self::FREQUENCY_ONE_TIME
                && $this->_values['Frequency'] != self::FREQUENCY_RECURRING
            ) {
                $specification->addValidationError(
                    'Frequency', 'Should be ' . self::FREQUENCY_ONE_TIME . ' or ' . self::FREQUENCY_RECURRING);
                return false;
            }

            if ($this->_values['TranType'] == self::TRAN_TYPE_VOICE_AUTH
                && !isset($this->_values['VoiceAuthCode'])
            ) {
                $specification->addValidationError(
                    'TranType', 'VoiceAuth no values VoiceAuthCode');
                return false;
            }

            if (isset($this->_values['AVSFields'])
                && ($this->_values['AVSFields'] != 'Off'
                && $this->_values['AVSFields'] != 'Zip'
                && $this->_values['AVSFields'] != 'Both')
            ) {
                $specification->addValidationError(
                    'AVSFields', 'Should be Off, Zip, Both');
                return false;
            }

            if (isset($this->_values['Keypad'])
                && ($this->_values['Keypad'] != 'On'
                && $this->_values['Keypad'] != 'Off')
            ) {
                $specification->addValidationError(
                    'Keypad', 'Should be On, Off');
                return false;
            }

            if (isset($this->_values['DefaultSwipe'])
                && $this->_values['DefaultSwipe'] != self::DEFAULT_SWIPE_MANUAL
                && $this->_values['DefaultSwipe'] != self::DEFAULT_SWIPE_SWIPE
            ) {
                $specification->addValidationError(
                    'DefaultSwipe', 'Should be ' . self::DEFAULT_SWIPE_MANUAL . ' or ' . self::DEFAULT_SWIPE_SWIPE);
                return false;
            }

            if (isset($this->_values['CardEntryMethod'])
                && ($this->_values['CardEntryMethod'] != 'Manual'
                && $this->_values['CardEntryMethod'] != 'Swipe'
                && $this->_values['CardEntryMethod'] != 'Both')
            ) {
                $specification->addValidationError(
                    'CardEntryMethod', 'Should be Manual, Swipe, Both');
                return false;
            }

            if (isset($this->_values['PageTimeoutDuration'])
                && ($this->_values['PageTimeoutDuration'] < 0
                || $this->_values['PageTimeoutDuration'] > 15)
            ) {
                $specification->addValidationError(
                    'PageTimeoutDuration', 'Should be between 0 and 15');
                return false;
            }

            if (isset($this->_values['PageTimeoutIndicator'])
                && ($this->_values['PageTimeoutIndicator'] != 'on'
                && $this->_values['PageTimeoutIndicator'] != 'off')
            ) {
                $specification->addValidationError(
                    'PageTimeoutIndicator', 'Should be on, off');
                return false;
            }

            if (isset($this->_values['DisplayStyle'])
                && ($this->_values['DisplayStyle'] != 'Mercury'
                && $this->_values['DisplayStyle'] != 'Custom')
            ) {
                $specification->addValidationError(
                    'DisplayStyle', 'Should be Mercury, Custom');
                return false;
            }

            if (isset($this->_values['FontFamily'])
                && ($this->_values['FontFamily'] != 'FontFamily1'
                && $this->_values['FontFamily'] != 'FontFamily2'
                && $this->_values['FontFamily'] != 'FontFamily3')
            ) {
                $specification->addValidationError(
                    'FontFamily', 'Should be FontFamily1, FontFamily2, FontFamily3');
                return false;
            }

            if (isset($this->_values['FontSize'])
                && ($this->_values['FontSize'] != 'Small'
                && $this->_values['FontSize'] != 'Medium'
                && $this->_values['FontSize'] != 'Large')
            ) {
                $specification->addValidationError(
                    'FontSize', 'Should be Small, Medium, Large');
                return false;
            }

            if (isset($this->_values['OrderTotal'])
                && ($this->_values['OrderTotal'] != 'On'
                && $this->_values['OrderTotal'] != 'Off')
            ) {
                $specification->addValidationError(
                    'OrderTotal', 'Should be On, Off');
                return false;
            }

            if (isset($this->_values['CancelButton'])
                && ($this->_values['CancelButton'] != 'on'
                && $this->_values['CancelButton'] != 'off')
            ) {
                $specification->addValidationError(
                    'CancelButton', 'Should be on, off');
                return false;
            }

            if (isset($this->_values['JCB'])
                && ($this->_values['JCB'] != 'on'
                    && $this->_values['JCB'] != 'off')
            ) {
                $specification->addValidationError(
                    'JCB', 'Should be on, off');
                return false;
            }

            if (isset($this->_values['Diners'])
                && ($this->_values['Diners'] != 'on'
                    && $this->_values['Diners'] != 'off')
            ) {
                $specification->addValidationError(
                    'Diners', 'Should be on, off');
                return false;
            }

            return true;

        } else {
            return false;
        }
    }

    /**
     * @return \stdClass|null
     */
    protected function _request()
    {

        $soapClient = new \MercuryHostedCheckoutService();
        $response = $soapClient->sendInitializePayment($this->_values);
        return (isset($response->InitializePaymentResult) ?
            $response->InitializePaymentResult : null);

    }

    /**
     *
     */
    protected function _requestSuccessProcess()
    {

        $response = $this->getResponse();
        $values = $this->getValues();

        $id = $values['Invoice'];

        $paymentId = $response->PaymentID;
        $responseCode = $response->ResponseCode;
        $responseMessage = $response->Message;
        $responseJson = json_encode($response);

        $mercuryModel = \Mercury::getInstance();

        $mercuryModel->updatePayment(
            $id,
            $mercuryModel::REQUEST_STATUS_INITIATED_SUCCESS,
            null,
            null,
            null,
            $paymentId);

        $mercuryModel->addRequest(
            $id,
            $mercuryModel::REQUEST_TYPE_INITIATE_PAYMENT,
            $responseCode,
            $responseMessage,
            $responseJson);

    }

    /**
     *
     */
    protected function _requestFailureProcess()
    {

        $response = $this->getResponse();
        $values = $this->getValues();

        $paymentId = $values['Invoice'];

        $responseCode = (isset($response->ResponseCode) ? $response->ResponseCode : null);
        $responseMessage = (isset($response->Message) ? $response->Message : null);
        $responseJson = json_encode($response);

        $mercuryModel = \Mercury::getInstance();

        $mercuryModel->updatePayment(
            $paymentId, $mercuryModel::REQUEST_STATUS_INITIATED_FAILURE);

        $mercuryModel->addRequest(
            $paymentId,
            $mercuryModel::REQUEST_TYPE_INITIATE_PAYMENT,
            $responseCode,
            $responseMessage,
            $responseJson);

    }

}
