<?php
/**
 * @package Mercury Payment Integration
 */


/**
 *
 */
class MercuryTest extends PHPUnit_Framework_TestCase
{

    /**
     * @var \CI_Controller
     */
    protected $CI;

    /**
     * @var \Mercury
     */
    protected $_object;

    /**
     *
     */
    public function setUp()
    {

        $this->CI = &get_instance();
        $this->CI->load->model('mercury');
        $this->CI->load->model('sale');

        /** @var \Mercury */
        $this->_object = & $this->CI->mercury;

    }

    /**
     *
     */
    public function testGetNextSaleId()
    {

        /** @var Mercury $mercuryModel */
        $mercuryModel = & $this->CI->mercury;

        $response = $mercuryModel->getNextSaleId();

        self::assertTrue(isset($response),
            'getNextSaleId() returned value (' . $response . ')');

    }

    /**
     *
     */
    public function ListTransactionServiceSoapMethods()
    {

        $mercuryClient = new \MercuryTransactionService();
        $soapClient = $mercuryClient::getSoapClient();
        $methods = get_class_methods($soapClient);

        if (method_exists($soapClient, '__getFunctions')) {

            $functions = $soapClient->__getFunctions();

            self::assertTrue(
                isset($functions) && is_array($functions) && sizeof($functions) > 0,
                'MercuryTransactionService methods: "' . print_r($functions, true) . '"');


        }

    }

    /**
     *
     */
    public function ListHostedCheckoutServiceSoapMethods()
    {

        $mercuryClient = new \MercuryHostedCheckoutService();
        $soapClient = $mercuryClient::getSoapClient();
        $methods = get_class_methods($soapClient);

        if (method_exists($soapClient, '__getFunctions')) {

            $functions = $soapClient->__getFunctions();

            self::assertTrue(
                isset($functions) && is_array($functions) && sizeof($functions) > 0,
                'MercuryHostedCheckoutService methods: "' . print_r($functions, true) . '"');


        }

    }


    /**
     *
     */
    public function SendCreditReturnTokenRequest()
    {

        if ($payments = $this->_object->getUnavailablePayments()) {

            foreach ($payments as $payment)
            {
                if ($request = $this->_object->getVerifiedRequestById(
                    $payment['request_payment_id']))
                {
                    if ($requestJson = json_decode($request['request_response_json'], true))
                    {
                        if (!empty($requestJson['Invoice'])
                            && !empty($requestJson['AuthAmount'])
                            && !empty($requestJson['Token'])
                        ) {
                            if ($request = $this->_object->creditReturnToken(
                                (int) $requestJson['Invoice'],
                                (double) $requestJson['AuthAmount'],
                                $requestJson['Token'])
                            ) {

                                $request->request();
                                $wasSuccessFul = $request->responseWasSuccessful();

                                self::assertTrue(
                                    !empty($wasSuccessFul),
                                    'CreditReturnToken request was successful');

                            }
                        } else {
                            self::display(
                                'Request JSON "' . print_r($requestJson, true) . '" lacks required fields');
                        }
                    }
                }
            }
        }

    }

    /**
     *
     */
    public function SendCreditVoidSaleTokenRequest()
    {

        if ($payments = $this->_object->getUnavailablePayments()) {

            foreach ($payments as $payment)
            {
                if ($request = $this->_object->getVerifiedRequestById(
                    $payment['request_payment_id']))
                {
                    if ($requestJson = json_decode($request['request_response_json'], true))
                    {
                        if (!empty($requestJson['AuthCode'])
                            && !empty($requestJson['Invoice'])
                            && !empty($requestJson['AuthAmount'])
                            && !empty($requestJson['RefNo'])
                            && !empty($requestJson['Token'])
                        ) {
                            if ($request = $this->_object->creditVoidSaleToken(
                                $requestJson['AuthCode'],
                                (int) $requestJson['Invoice'],
                                (double) $requestJson['AuthAmount'],
                                $requestJson['RefNo'],
                                $requestJson['Token'])
                            ) {

                                $request->request();
                                $wasSuccessFul = $request->responseWasSuccessful();

                                self::assertTrue(
                                    !empty($wasSuccessFul),
                                    'CreditVoidSaleToken request was successful');

                            }
                        } else {
                            self::display(
                                'Request JSON "' . print_r($requestJson, true) . '" lacks required fields');
                        }
                    }
                }
            }

        }

    }

    /**
     *
     */
    public function testGetRequestsForVerification()
    {

        $requests = $this->_object->getRequestsForVerification();

        self::assertTrue(
            isset($requests) && (is_array($requests) || $requests === false),
            'getRequestsForVerification() returned array of false');

        if ($requests) {
            self::display('getRequestsForVerification() returned array "' . print_r($requests, true) . '"');
        }

    }

    /**
     *
     */
    public function testVerifyPaymentRequest()
    {

        if ($requests = $this->_object->getRequestsForVerification()) {

            foreach ($requests as $request)
            {

                $request =
                    $this->_object->verifyPayment($request['request_payment_paymentid']);
                $request->request();

                $wasSuccessful = $request->responseWasSuccessful();

                self::assertTrue(
                    !empty($wasSuccessful),
                    'VerifyPayment request was successful');

                if ($wasSuccessful) {
                    self::display('Succeeded with response: ' . print_r($request->getResponse(), true));
                } else {
                    self::display('Failed with response: ' . print_r($request->getResponse(), true));
                }


            }

        }

    }

    /**
     * @throws \Exception
     */
    public function testProcessSalePayments()
    {

        $paymentId = $this->_object->addPayment(
            \Mercury::REQUEST_STATUS_VERIFIED_PAYMENT_SUCCESS,
            'Credit Card',
            5,
            'Oek0EKoeJAOJ2O',
            'koekofkeo2k1030123o'
        );

        self::assertTrue(
            isset($paymentId)
            && is_numeric($paymentId),
            'Added payment id ' . $paymentId,
            true);

        $paymentIds[] = $paymentId;
        $payments[] = array(
            'payment_type' => 'Credit Card',
            'payment_amount' => 5,
        );

        $paymentId = $this->_object->addPayment(
            \Mercury::REQUEST_STATUS_VERIFIED_PAYMENT_SUCCESS,
            'Debit Card',
            9,
            'O31093JEOAK2AeeOKE',
            'ko3k1023oa3i'
        );

        self::assertTrue(
            isset($paymentId)
            && is_numeric($paymentId),
            'Added payment id ' . $paymentId,
            true);

        $paymentIds[] = $paymentId;
        $payments[] = array(
            'payment_type' => 'Debit Card',
            'payment_amount' => 9,
        );

        $saleId = $this->_object->getNextSaleId();

        self::assertTrue(
            isset($saleId)
            && is_numeric($saleId),
            'Got next sale id ' . $saleId,
            true);

        $this->_object->processSalePayments(
            $saleId,
            $payments,
            true);

        foreach ($paymentIds as $paymentId)
        {

            $payment = $this->_object->getPaymentById($paymentId);

            self::assertTrue(
                isset($payment)
                && is_array($payment)
                && isset($payment['request_payment_sale_id'])
                && $payment['request_payment_sale_id'] == $saleId,
                'Payment ' . $paymentId . ' has updated sale id ' . $saleId . ' for generated sale.',
                true);

        }

        foreach ($paymentIds as $paymentId)
        {
            $response = $this->_object->deletePayment($paymentId);
            self::assertTrue(
                !empty($response),
                'Deleted payment ' . $paymentId,
                true);
        }

    }

    /**
     *
     */
    public function testRemoveCurrencyFromAmount()
    {

        $amount = '50.0';
        $amountCurrency = '$' . $amount;

        $response = $this->_object->removeCurrencyFromAmount($amountCurrency);

        self::assertTrue(
            isset($response)
            && $response === $amount,
            'Returned amount "'. $response . '" matches expected amount "' . $amount . '" from "' . $amountCurrency . '"',
            true);

        $amount = '5.50';
        $amountCurrency = $amount . ' SEK';

        $response = $this->_object->removeCurrencyFromAmount($amountCurrency);

        self::assertTrue(
            isset($response)
            && $response === $amount,
            'Returned amount "'. $response . '" matches expected amount "' . $amount . '" from "' . $amountCurrency . '"',
            true);


    }

    /**
     * This method should:
     * - Create fake simulated payment requests for type and amount.
     * - Create a fake sale
     * - Test if processPayments connect the two successfully.
     *
     * @throws \Exception
     */
    public function SaleComplete()
    {

        $paymentIds = array();
        $payments = array();

        $employee = $this->_object->getFirstEmployee();
        self::assertTrue(
            isset($employee)
            && is_array($employee)
            && sizeof($employee) > 0,
            'Found a first employee',
            true);


        $customer = $this->_object->getFirstCustomer();
        self::assertTrue(
            isset($customer)
            && is_array($customer)
            && sizeof($customer) > 0,
            'Found a first customer',
            true);

        $item = $this->_object->getFirstItem();
        self::assertTrue(
            isset($item)
            && is_array($item)
            && sizeof($item) > 0,
            'Found a first item',
            true);

        $item['line'] = 1;
        $item['quantity'] = 1;
        $item['discount'] = 0;
        $item['price'] = $item['unit_price'];
        $item['serialnumber'] = '';

        $paymentId = $this->_object->addPayment(
            \Mercury::REQUEST_STATUS_VERIFIED_PAYMENT_SUCCESS,
            'Credit Card',
            5,
            'Oek0EKoeJAOJ2O',
            'koekofkeo2k1030123o'
        );

        self::assertTrue(
            isset($paymentId)
            && is_numeric($paymentId),
            'Added payment id ' . $paymentId,
            true);

        $paymentIds[] = $paymentId;
        $payments[] = array(
            'payment_type' => 'Credit Card',
            'payment_amount' => 5,
        );

        $paymentId = $this->_object->addPayment(
            \Mercury::REQUEST_STATUS_VERIFIED_PAYMENT_SUCCESS,
            'Debit Card',
            9,
            'O31093JEOAK2AeeOKE',
            'ko3k1023oa3i'
        );

        self::assertTrue(
            isset($paymentId)
            && is_numeric($paymentId),
            'Added payment id ' . $paymentId,
            true);

        $paymentIds[] = $paymentId;
        $payments[] = array(
            'payment_type' => 'Debit Card',
            'payment_amount' => 9,
        );

        $items = array($item);

        /** @var \Sale $saleModel */
        $saleModel = & $this->CI->sale;

        $saleId = $saleModel->save(
            $items,
            $customer['person_id'],
            $employee['person_id'],
            '',
            $payments);

        self::assertTrue(
            isset($saleId)
            && is_numeric($saleId),
            'Saved sales saleId is numeric ' . $saleId,
            true);

        $this->_object->processSalePayments(
            $saleId,
            $payments,
            true);

        foreach ($paymentIds as $paymentId)
        {

            $payment = $this->_object->getPaymentById($paymentId);

            self::assertTrue(
                isset($payment)
                && is_array($payment)
                && isset($payment['request_payment_sale_id'])
                && $payment['request_payment_sale_id'] == $saleId,
                'Payment ' . $paymentId . ' has updated sale id ' . $saleId . ' for generated sale "'
                . print_r($payments, true) . '" ',
                true);

        }


    }

    /**
     *
     */
    public function GetPaymentsReceiptBySaleIdString()
    {

        if ($payments = $this->_object->getUnavailablePayments()) {

            $usedPaymentIds = array();

            foreach ($payments as $payment)
            {

                if (!isset($usedPaymentIds[$payment['request_payment_sale_id']])) {

                    $receipt = $this->_object->getPaymentsReceiptBySaleIdString(
                        'POS ' . $payment['request_payment_sale_id']);
                    self::assertTrue(
                        isset($receipt) && is_array($receipt) && sizeof($receipt) > 0,
                        'getPaymentsReceiptBySaleIdString(\'POS ' . $payment['request_payment_sale_id'] . '\') returned non-empty array "' . print_r($receipt, true) . '"',
                        true);

                    $usedPaymentIds[$payment['request_payment_sale_id']] = true;

                }

            }

        }

    }

    /**
     *
     */
    public function InitializePaymentRequest()
    {

        /** @var Mercury $mercuryModel */
        $mercuryModel = & $this->CI->mercury;

        $request = $mercuryModel->initializePayment('Credit card', 5);
        $request->request();

        $wasSuccessfull = $request->responseWasSuccessful();

        self::assertTrue(
            !empty($wasSuccessfull),
            'InitializePayment request was successful');

        if ($wasSuccessfull) {

            $response = $mercuryModel->displayHostedCheckoutRedirect($request);

            self::display('displayHostedCheckoutRedirect response: "' . $response . '"');

            self::assertTrue(
                !empty($response),
                'displayHostedCheckoutRedirect was successful');

            $response = $mercuryModel->displayHostedCheckoutIframe($request);

            self::display('displayHostedCheckoutIframe response: "' . $response . '"');

            self::assertTrue(
                !empty($response),
                'displayHostedCheckoutIframe was successful');


        } else {

            self::display('Failed with response: ' . print_r($request->getResponse(), true));

        }

    }

    /**
     *
     */
    public function testModelAvailable()
    {

        /** @var Mercury $mercuryModel */
        $mercuryModel = & $this->CI->mercury;

        $this->assertTrue(isset($mercuryModel), 'Mercury model available');

        $settings = $mercuryModel->getSettings();

        $this->assertTrue(isset($settings) && is_array($settings) && sizeof($settings) > 0,
            'Mercury settings available');

        // self::display('Settings:' . print_r($settings, true));

    }

    /**
     * @static
     * @param bool $condition
     * @param string [$message = '']
     * @param bool $displayOnSuccess
     */
    public static function assertTrue($condition, $message = '', $displayOnSuccess = true)
    {
        parent::assertTrue($condition, $message);
        if (!empty($condition)) {
            if ($displayOnSuccess) {
                self::display('SUCEEDED - "' . $message . '"');
            }
        } else {
            self::display('FAIL - "' . $message . '"');
        }
    }

    /**
     * @static
     * @param string [$message = '']
     */
    public static function display($message = '')
    {
        if (!empty($message)) {
            echo "\n" . $message . "\n";
        }
    }

}

