<?php
/**
 * @package Mercury Payment Integration
 */

/**
 *
 */
class MercuryTransactionService
{

    /**
     * @var string
     */
    const DEVELOPMENT_URL = 'https://hc.mercurydev.net/tws/transactionservice.asmx';

    /**
     * @var string
     */
    const PRODUCTION_URL = 'https://hc.mercurypay.com/tws/transactionservice.asmx';

    /**
     * @static
     * @var \MercuryTransactionServiceInterface
     */
    private static $_wsClient;

    /**
     *
     */
    public function __construct()
    {
        self::_init();
    }

    /**
     * @internal
     * @static
     */
    private static function _init()
    {
        if (!isset(self::$_wsClient)) {
            if (self::_isInDevelopment()) {
                $url = self::DEVELOPMENT_URL . '?WSDL';
            } else {
                $url = self::PRODUCTION_URL . '?WSDL';
            }

            self::$_wsClient = new SoapClient($url);

        }
    }

    /**
     * @static
     * @return bool
     */
    private static function _isInDevelopment()
    {
        return \Mercury::isTesting();
    }

    /**
     * @static
     * @return \MercuryTransactionServiceInterface
     */
    public static function getSoapClient()
    {
        self::_init();
        return self::$_wsClient;
    }

    /**
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=BatchClear
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendBatchClear($request, $password)
    {
        self::_init();
        return self::$_wsClient->BatchClear(array(
            'request' => $request, 'password' => $password));
    }

    /**
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=BatchClose
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendBatchClose($request, $password)
    {
        self::_init();
        return self::$_wsClient->BatchClose(array(
            'request' => $request, 'password' => $password));
    }

    /**
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=BatchSummary
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendBatchSummary($request, $password)
    {
        self::_init();
        return self::$_wsClient->BatchSummary(array(
            'request' => $request, 'password' => $password));
    }

    /**
     * For adjusting the amount of the original Sale or PreAuthCapture.
     *
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=CreditAdjustToken
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendCreditAdjustToken($request, $password)
    {
        self::_init();
        return self::$_wsClient->CreditAdjustToken(array(
            'request' => $request, 'password' => $password));
    }

    /**
     * For finalizing a PreAuth.
     *
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=CreditPreAuthCaptureToken
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendCreditPreAuthCaptureToken($request, $password)
    {
        self::_init();
        return self::$_wsClient->CreditPreAuthCaptureToken(array(
            'request' => $request, 'password' => $password));
    }

    /**
     * For Recurring billing and card-on-file transactions.
     *
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=CreditPreAuthToken
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendCreditPreAuthToken($request, $password)
    {
        self::_init();
        return self::$_wsClient->CreditPreAuthToken(array(
            'request' => $request, 'password' => $password));
    }

    /**
     * For returning a previously ran transaction.
     *
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=CreditReturnToken
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendCreditReturnToken($request, $password)
    {
        self::_init();
        return self::$_wsClient->CreditReturnToken(array(
            'request' => $request, 'password' => $password));
    }

    /**
     * For reversing PreAuth, PreauthCapture, and Sale transactions.
     *
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=CreditReversalToken
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendCreditReversalToken($request, $password)
    {
        self::_init();
        return self::$_wsClient->CreditReversalToken(array(
            'request' => $request, 'password' => $password));
    }

    /**
     * For Recurring billing and card-on-file transactions.
     *
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=CreditSaleToken
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendCreditSaleToken($request, $password)
    {
        self::_init();
        return self::$_wsClient->CreditSaleToken(array(
            'request' => $request, 'password' => $password));
    }

    /**
     * For voiding a Return transaction in the current batch.
     *
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=CreditVoidReturnToken
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendCreditVoidReturnToken($request, $password)
    {
        self::_init();
        return self::$_wsClient->CreditVoidReturnToken(array(
            'request' => $request, 'password' => $password));
    }

    /**
     * For voiding a Sale or PreAuthCapture transaction in the current batch.
     *
     * @see https://hc.mercurydev.net/tws/transactionservice.asmx?op=CreditVoidSaleToken
     * @param array $request
     * @param string $password
     * @return \stdClass
     */
    public function sendCreditVoidSaleToken($request, $password)
    {
        self::_init();
        return self::$_wsClient->CreditVoidReturnToken(array(
            'request' => $request, 'password' => $password));
    }

}
