<?php
/**
 * @package Mercury Payment Integration
 */

require_once(__DIR__ . '/mercury/requests/MercuryRequest.php');
require_once(__DIR__ . '/mercury/requests/MercuryRequestSpecification.php');
require_once(__DIR__ . '/mercury/requests/MercuryRequestSpecificationParameter.php');

require_once(__DIR__ . '/mercury/requests/CreditReturnTokenRequest.php');
require_once(__DIR__ . '/mercury/requests/CreditVoidReturnTokenRequest.php');
require_once(__DIR__ . '/mercury/requests/CreditVoidSaleTokenRequest.php');
require_once(__DIR__ . '/mercury/requests/HostedCheckoutRequest.php');
require_once(__DIR__ . '/mercury/requests/InitializePaymentRequest.php');
require_once(__DIR__ . '/mercury/requests/VerifyPaymentRequest.php');

require_once(__DIR__ . '/mercury/soap_clients/MercuryHostedCheckoutService.php');
require_once(__DIR__ . '/mercury/soap_clients/MercuryTransactionService.php');

/**
 *
 */
class Mercury extends CI_Model
{

    /**
     * @var bool
     */
    private static $_testing;

    /**
     * @var string
     */
    const MODULE_ID = 'PaymentGateway';

    /**
     * @var string
     */
    const TABLE_SETTINGS = 'paymentgateway_settings';

    /**
     * @var string
     */
    const TABLE_REQUESTS = 'paymentgateway_requests';

    /*
     * @var string
     */
    const TABLE_PAYMENTS = 'paymentgateway_payments';

    /**
     * @var int
     */
    const REQUEST_STATUS_INITIATED_SUCCESS = 0;

    /**
     * @var int
     */
    const REQUEST_STATUS_INITIATED_FAILURE = 1;

    /**
     * @var int
     */
    const REQUEST_STATUS_HOSTED_CHECKOUT_SUCCESS = 2;

    /**
     * @var int
     */
    const REQUEST_STATUS_HOSTED_CHECKOUT_FAILURE = 3;

    /**
     * @var int
     */
    const REQUEST_STATUS_VERIFIED_PAYMENT_SUCCESS = 4;

    /**
     * @var int
     */
    const REQUEST_STATUS_VERIFIED_PAYMENT_FAILURE = 5;

    /**
     * @var int
     */
    const REQUEST_STATUS_RETURN_SALE_FAILURE = 6;

    /**
     * @var int
     */
    const REQUEST_STATUS_RETURN_SALE_SUCCESS = 7;

    /**
     * @var int
     */
    const REQUEST_STATUS_VOID_SALE_FAILURE = 8;

    /**
     * @var int
     */
    const REQUEST_STATUS_VOID_SALE_SUCCESS = 9;

    /**
     * @var int
     */
    const REQUEST_STATUS_CREATED = 10;

    /**
     * @var int
     */
    const REQUEST_STATUS_VOID_RETURN_FAILURE = 11;

    /**
     * @var int
     */
    const REQUEST_STATUS_VOID_RETURN_SUCCESS = 12;

    /**
     * @var int
     */
    const REQUEST_TYPE_INITIATE_PAYMENT = 0;

    /**
     * @var int
     */
    const REQUEST_TYPE_HOSTED_CHECKOUT = 1;

    /**
     * @var int
     */
    const REQUEST_TYPE_VERIFY_PAYMENT = 2;

    /**
     * @var int
     */
    const REQUEST_TYPE_RETURN_TOKEN = 3;

    /**
     * @var int
     */
    const REQUEST_TYPE_VOID_RETURN_TOKEN = 4;

    /**
     * @var int
     */
    const REQUEST_TYPE_VOID_SALE_TOKEN = 5;

    /**
     * @var string
     */
    const MEMO_TAG = 'SaberPHP';

    /**
     * @var \Mercury
     */
    private static $_instance;

    /**
     * @var \CI_Controller
     */
    private $_CI;

    /**
     * @throws \Exception
     */
    public function __construct()
    {

        parent::__construct();
        $this->load->dbforge();
        $this->_CI = & get_instance();

        if (!$this->_isInstalled()) {
            if (!$this->_install() ) {
                Throw new Exception(
                    'Failed to install ' . __CLASS__
                    . ' in ' . __FILE__);
            }
        }

        $settings = $this->getSettings();
        $testing = (!empty($settings['testing_mode']));
        self::setTesting($testing);

        self::$_instance = $this;

    }

    /**
     * @param string $amount
     * @return string
     */
    public function formatAmount($amount)
    {
        if (isset($amount)) {
            if ($amount >= 0) {
                return number_format($amount, 2, '.', '');
            } else {
                return '-' . number_format(abs($amount), 2, '.', '');
            }
        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;
    }

    /**
     * @param array $payments
     * @param bool|null [$throwExceptionFail = null]
     */
    public function preProcessSalePayments($payments, $throwExceptionFail = null)
    {
        foreach ($payments as $payment)
        {
            if (!$mercuryPayment = $this->getAvailablePaymentByTypeAndAmount(
                $payment['payment_type'], $payment['payment_amount'])
            ) {
                if ($payment['payment_type'] == 'Credit Card'
                    || $payment['payment_type'] == 'Debit Card'
                    || !empty($throwExceptionFail)
                ) {
                    die(
                        'Failed to find verified card payment. '
                        . 'Type: "' . $payment['payment_type'] . '", Amount: "' . $payment['payment_amount'] . '".'
                        . ' Please go back and verify all card payments before completing sale.');
                }
            }
        }
    }

    /**
     * @param int $saleId
     * @param array $payments
     * @param bool|null [$throwExceptionFail = null]
     * @throws \Exception
     */
    public function processSalePayments($saleId, $payments, $throwExceptionFail = null)
    {
        foreach ($payments as $payment)
        {
            if (!$mercuryPayment = $this->getAvailablePaymentByTypeAndAmount(
                $payment['payment_type'], $payment['payment_amount'])
            ) {
                if ($payment['payment_type'] == 'Credit Card'
                    || $payment['payment_type'] == 'Debit Card'
                    || !empty($throwExceptionFail)
                ) {
                    Throw new \Exception(
                        'Failed to find verified card payment. '
                         . 'Type: "' . $payment['payment_type'] . '", Amount: "' . $payment['payment_amount'] . '".'
                         . ' Please go back and verify all card payments before completing sale.');
                }
            }
        }
        reset($payments);
        foreach ($payments as $payment)
        {
            if ($mercuryPayment = $this->getAvailablePaymentByTypeAndAmount(
                $payment['payment_type'], $payment['payment_amount'])
            ) {
                $this->updatePayment(
                    $mercuryPayment['request_payment_id'],
                    null,
                    null,
                    null,
                    null,
                    null,
                    $saleId);
            } else {
                if ($payment['payment_type'] == 'Credit Card'
                    || $payment['payment_type'] == 'Debit Card'
                    || !empty($throwExceptionFail)
                ) {
                    Throw new \Exception(
                        'Failed to find verified card payment. '
                        . 'Type: "' . $payment['payment_type'] . '", Amount: "' . $payment['payment_amount'] . '".'
                        . ' Please go back and verify all card payments before completing sale.');
                }
            }
        }
    }

    /**
     * @param string $amount
     * @return string
     */
    public function removeCurrencyFromAmount($amount)
    {
        return preg_replace('/[^\d\.\-]/', '', $amount);
    }

    /**
     * @return array|bool
     */
    public function getFirstItem()
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->from('items');
        $db->order_by('item_id', 'DESC');
        $db->limit(1);

        if ($query = $db->get()) {
            if ($results = $query->result('array')) {
                return $results[0];
            }
        }

        return false;

    }

    /**
     * @return array|bool
     */
    public function getFirstCustomer()
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->from('customers');
        $db->order_by('person_id', 'DESC');
        $db->limit(1);

        if ($query = $db->get()) {
            if ($results = $query->result('array')) {
                return $results[0];
            }
        }

        return false;

    }

    /**
     * @return array|bool
     */
    public function getFirstEmployee()
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->from('employees');
        $db->order_by('person_id', 'DESC');
        $db->limit(1);

        if ($query = $db->get()) {
            if ($results = $query->result('array')) {
                return $results[0];
            }
        }

        return false;

    }


    /**
     * @param string $type
     * @param string $amount
     * @return array|bool
     */
    public function getAvailablePaymentByTypeAndAmount($type, $amount)
    {

        if (isset($type, $amount)) {

            $amount = $this->formatAmount($amount);

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;
            $db->from(self::TABLE_PAYMENTS);
            $db->where('request_payment_type', $type);
            $db->where('request_payment_amount', $amount);
            $db->where('request_payment_status', self::REQUEST_STATUS_VERIFIED_PAYMENT_SUCCESS);
            $db->where('request_payment_sale_id IS NULL');
            $db->limit(1);

            if ($query = $db->get()) {
                if ($results = $query->result('array')) {
                    return $results[0];
                }
            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;

    }

    /**
     * @param string $type
     * @param string $amount
     * @return array|bool
     */
    public function getUnverifiedPaymentByTypeAndAmount($type, $amount)
    {

        if (isset($type, $amount)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;
            $db->from(self::TABLE_PAYMENTS);
            $db->where('request_payment_type', $type);
            $db->where('request_payment_amount', $amount);
            $db->where('request_payment_status IN ('
                . self::REQUEST_STATUS_CREATED
                . ',' . self::REQUEST_STATUS_INITIATED_FAILURE . ')');
            $db->where('request_payment_sale_id IS NULL');
            $db->limit(1);

            if ($query = $db->get()) {
                if ($results = $query->result('array')) {
                    return $results[0];
                }
            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;

    }

    /**
     * @param string $type
     * @param string $amount
     * @return array|bool
     */
    public function getInitiatedPaymentByTypeAndAmount($type, $amount)
    {

        if (isset($type, $amount)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;
            $db->from(self::TABLE_PAYMENTS);
            $db->where('request_payment_type', $type);
            $db->where('request_payment_amount', $amount);
            $db->where('request_payment_status', self::REQUEST_STATUS_INITIATED_SUCCESS);
            $db->where('request_payment_sale_id IS NULL');
            $db->where('request_payment_paymentid IS NOT NULL');
            $db->limit(1);

            if ($query = $db->get()) {
                if ($results = $query->result('array')) {
                    return $results[0];
                }
            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;

    }

    /**
     * @return array|bool
     */
    public function getUnavailablePayments()
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->from(self::TABLE_PAYMENTS);
        $db->where('request_payment_status', self::REQUEST_STATUS_VERIFIED_PAYMENT_SUCCESS);
        $db->where('request_payment_sale_id IS NOT NULL');

        if ($query = $db->get()) {
            if ($results = $query->result('array')) {
                return $results;
            }
        }

        return false;

    }

    /**
     * @param int $saleId
     * @return array|bool
     */
    public function getPaymentsBySaleId($saleId)
    {
        if (isset($saleId)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;
            $db->from(self::TABLE_PAYMENTS);
            $db->where('request_payment_sale_id', $saleId);

            if ($query = $db->get()) {
                if ($results = $query->result('array')) {
                    return $results;
                }
            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;
    }

    /**
     * @param int $id
     * @return array|bool
     */
    public function getReturnedRequestById($id)
    {
        if (isset($id)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;
            $db->from(self::TABLE_REQUESTS);
            $db->where('request_payment_id', $id);
            $db->where('request_type', self::REQUEST_TYPE_RETURN_TOKEN);
            $db->order_by('request_added',  'DESC');
            $db->limit(1);

            if ($query = $db->get()) {
                if ($results = $query->result('array')) {
                    if (isset($results[0])) {
                        return $results[0];
                    }
                }
            }
            return false;

        } else {
        }
        return false;
    }

    /**
     * @param int $id
     * @return array|bool
     */
    public function getVerifiedRequestById($id)
    {
        if (isset($id)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;
            $db->from(self::TABLE_REQUESTS);
            $db->where('request_payment_id', $id);
            $db->where('request_type', self::REQUEST_TYPE_VERIFY_PAYMENT);
            $db->limit(1);

            if ($query = $db->get()) {
                if ($results = $query->result('array')) {
                    if (isset($results[0])) {
                        return $results[0];
                    }
                }
            }
            return false;

        } else {
        }
        return false;
    }

    /**
     * @return array
     */
    public function getStatuses()
    {
        return array(
            self::REQUEST_STATUS_INITIATED_FAILURE => 'Initiate payment failure',
            self::REQUEST_STATUS_INITIATED_SUCCESS => 'Initiate payment success',
            self::REQUEST_STATUS_HOSTED_CHECKOUT_FAILURE => 'Hosted checkout failure',
            self::REQUEST_STATUS_HOSTED_CHECKOUT_SUCCESS => 'Hosted checkout success',
            self::REQUEST_STATUS_VERIFIED_PAYMENT_FAILURE => 'Verify payment failure',
            self::REQUEST_STATUS_VERIFIED_PAYMENT_SUCCESS => 'Verify payment success',
            self::REQUEST_STATUS_RETURN_SALE_FAILURE => 'Return sale failure',
            self::REQUEST_STATUS_RETURN_SALE_SUCCESS => 'Return sale success',
            self::REQUEST_STATUS_VOID_SALE_FAILURE => 'Void sale failure',
            self::REQUEST_STATUS_VOID_SALE_SUCCESS => 'Void sale success',
            self::REQUEST_STATUS_CREATED => 'Created',
            self::REQUEST_STATUS_VOID_RETURN_FAILURE => 'Void return sale failure',
            self::REQUEST_STATUS_VOID_RETURN_SUCCESS => 'Void return sale success',
        );
    }

    /**
     * @return array|bool
     */
    public function getVerifiedPayments()
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->from(self::TABLE_PAYMENTS);
        $db->where('request_payment_status', self::REQUEST_STATUS_VERIFIED_PAYMENT_SUCCESS);
        $db->where('request_payment_token IS NOT NULL');
        $db->order_by('request_payment_updated', 'DESC');

        if ($query = $db->get()) {
            if ($results = $query->result('array')) {
                return $results;
            }
        }
        return false;

    }

    /**
     * @param int $id
     * @return array|bool
     */
    public function getVerifiedPaymentById($id)
    {

        if (isset($id)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;
            $db->from(self::TABLE_PAYMENTS);
            $db->where('request_payment_status', self::REQUEST_STATUS_VERIFIED_PAYMENT_SUCCESS);
            $db->where('request_payment_token IS NOT NULL');
            $db->where('request_payment_id', $id);
            $db->limit(1);

            if ($query = $db->get()) {
                if ($results = $query->result('array')) {
                    if (isset($results[0])) {
                        return $results[0];
                    }
                }
            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;

    }

    /**
     * @param int $id
     * @return array|bool
     */
    public function getReturnedPaymentById($id)
    {

        if (isset($id)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;
            $db->from(self::TABLE_PAYMENTS);
            $db->where('request_payment_status', self::REQUEST_STATUS_RETURN_SALE_SUCCESS);
            $db->where('request_payment_token IS NOT NULL');
            $db->where('request_payment_id', $id);
            $db->limit(1);

            if ($query = $db->get()) {
                if ($results = $query->result('array')) {
                    if (isset($results[0])) {
                        return $results[0];
                    }
                }
            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;

    }

    /**
     * @param int|null [$start = null]
     * @param int|null [$limit = null]
     * @param string|null [$searchFilter = null]
     * @return array|bool
     */
    public function getPayments(
        $start = null, $limit = null, $searchFilter = null)
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->from(self::TABLE_PAYMENTS);
        if (isset($start, $limit)) {
            $db->limit($limit, $start);
        }
        if (isset($searchFilter)) {
            $db->where(
                '(request_payment_amount LIKE "%' . $db->escape_str($searchFilter) . '%" '
                . 'OR request_payment_type LIKE "%' . $db->escape_str($searchFilter) . '%" '
                . 'OR request_payment_sale_id LIKE "%' . $db->escape_str($searchFilter) . '%" '
                . 'OR request_payment_status LIKE "%' . $db->escape_str($searchFilter) . '%" '
                . 'OR request_payment_id LIKE "%' . $db->escape_str($searchFilter) . '%")'
            );
        }
        $db->order_by('request_payment_updated', 'DESC');

        if ($query = $db->get()) {
            if ($results = $query->result('array')) {
                return $results;
            }
        }
        return false;

    }

    /**
     * @param string|null [$searchFilter = null]
     * @return int
     */
    public function getPaymentsCount($searchFilter = null)
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->select('COUNT(*) as count');
        $db->from(self::TABLE_PAYMENTS);
        if (isset($searchFilter)) {
            $db->where(
                '(request_payment_amount LIKE "%' . $db->escape_str($searchFilter) . '%" '
                . 'OR request_payment_type LIKE "%' . $db->escape_str($searchFilter) . '%" '
                . 'OR request_payment_sale_id LIKE "%' . $db->escape_str($searchFilter) . '%" '
                . 'OR request_payment_status LIKE "%' . $db->escape_str($searchFilter) . '%")'
            );
        }


        if ($query = $db->get()) {
            if ($results = $query->result('array')) {
                if (isset($results[0]['count'])) {
                    return $results[0]['count'];
                }
            }
        }
        return 0;

    }

    /**
     * @param string $saleIdString
     * @return array|bool
     */
    public function getPaymentsReceiptBySaleIdString($saleIdString)
    {
        if (!empty($saleIdString)) {

            $saleId = (int) str_replace('POS ', '', $saleIdString);
            if ($payments = $this->getPaymentsBySaleId($saleId))
            {

                $count = 0;
                $receipt = array();

                foreach ($payments as $payment)
                {
                    if ($request = $this->getVerifiedRequestById(
                        $payment['request_payment_id']))
                    {
                        if ($requestJson = json_decode($request['request_response_json'], true))
                        {
                            if (!empty($requestJson['CardType'])
                                && !empty($requestJson['MaskedAccount'])
                                && !empty($requestJson['Amount'])
                                && !empty($requestJson['TransPostTime'])
                                && !empty($requestJson['ExpDate'])
                            ) {
                                $receipt[] =
                                    $requestJson['CardType']
                                    . ' ' . $requestJson['MaskedAccount']
                                    . ' ' . $requestJson['ExpDate']
                                    . '<br />' . $requestJson['TransPostTime']
                                    . ' ' . to_currency($requestJson['Amount'])
                                    . '<br />&nbsp;<br />&nbsp;<br />Signature:&nbsp;&nbsp;&nbsp;&nbsp;_____________________________';
                                $count++;
                            }
                        }
                    }
                }

                if (sizeof($receipt) > 0) {
                    return $receipt;
                } else {
                    return false;
                }

            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;
    }

    /**
     * @param string $type
     * @param string $amount
     * @return bool
     */
    public function availablePaymentExistsByTypeAndAmount($type, $amount)
    {
        if ($this->getAvailablePaymentByTypeAndAmount($type, $amount)) {
            return true;
        }
        return false;
    }

    /**
     * @return \Mercury
     */
    public static function getInstance()
    {
        if (!isset(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * @param bool $testing
     */
    public static function setTesting($testing)
    {
        self::$_testing = (!empty($testing));
    }

    /**
     * @return bool
     */
    public static function isTesting()
    {
        return (!empty(self::$_testing));
    }

    /**
     * @return array
     */
    public function getSessionPayments()
    {
        return $this->_CI->session->userdata('payments');
    }

    /**
     * @return array
     */
    public function getSettings()
    {

        $query = $this->db->get(self::TABLE_SETTINGS);
        $settings = array();

        foreach ($query->result('array') as $row)
        {
            $settings[$row['setting_key']] = $row['setting_value'];
        }

        return $settings;

    }

    /**
     * @param string $key
     * @return string
     */
    public function getSetting($key)
    {
        if (!empty($key)) {
            $settings = $this->getSettings();
            if (isset($settings[$key])) {
                return $settings[$key];
            }
        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;
    }

    /**
     * @return int
     */
    public function getNextSaleId()
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->select_max('sale_id', 'max_sale_id');
        $db->from('sales');
        $query = $db->get();

        $results = $query->result('array');
        $result = $results[0];

        return (int) ($result['max_sale_id'] + 1);

    }

    /**
     * @return int
     */
    public function getNextPaymentId()
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->select_max('request_payment_id', 'max_payment_id');
        $db->from(self::TABLE_PAYMENTS);

        if ($query = $db->get()) {

            $results = $query->result('array');
            $result = $results[0];
            return (int) ($result['max_payment_id'] + 1);

        } else {
            return 0;
        }

    }

    /**
     * @param string|null $paymentId
     * @return array|bool
     */
    public function getPaymentByPaymentId($paymentId)
    {
        if (isset($paymentId)) {
            /** @var \CI_DB_active_record $db */
            $db = & $this->db;
            $db->where('request_payment_paymentid', $paymentId);
            $db->from(self::TABLE_PAYMENTS);
            $db->limit(1);

            if ($query = $db->get()) {
                if ($results = $query->result('array')) {
                    if (isset($results[0])) {
                        return $results[0];
                    }
                }
            }
        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;

    }

    /**
     * @param string|null $id
     * @return array|bool
     */
    public function getPaymentById($id)
    {
        if (isset($id)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;
            $db->where('request_payment_id', $id);
            $db->from(self::TABLE_PAYMENTS);
            $db->limit(1);

            if ($query = $db->get()) {
                if ($results = $query->result('array')) {
                    if (isset($results[0])) {
                        return $results[0];
                    }
                }
            }
        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;

    }

    /**
     * @return array|bool
     */
    public function getRequestsForVerification()
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->from(self::TABLE_PAYMENTS . ' AS payments');
        $db->where('request_payment_status', self::REQUEST_STATUS_HOSTED_CHECKOUT_SUCCESS);

        if ($query = $db->get()) {
            if ($results = $query->result('array')) {
                return $results;
            }
        }
        return false;

    }

    /**
     * @return int|bool
     */
    public function getCurrentPersonId()
    {
        return $this->session->userdata('person_id');
    }

    /**
     * @return bool
     */
    public function isLoggedIn()
    {
        return ($this->session->userdata('person_id') !== false);
    }

    /**
     * @param array $arrayKeysToValues
     * @return bool
     */
    public function setSettings($arrayKeysToValues)
    {
        if (isset($arrayKeysToValues)
            && is_array($arrayKeysToValues)
            && sizeof($arrayKeysToValues) > 0
        ) {

            $this->db->truncate(self::TABLE_SETTINGS);
            $accBool = true;

            foreach ($arrayKeysToValues as $key => $value)
            {
                $accBool = ($accBool &&
                $this->db->insert(self::TABLE_SETTINGS, array(
                    'setting_key' => $key,
                    'setting_value' => $value,
                )));
            }

            return $accBool;

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;
    }

    /**
     * @param string $paymentId
     * @param int|null [$type = null]
     * @param string|null [$responseCode = null]
     * @param string|null [$responseMessage = null]
     * @param string|null [$responseJson = null]
     * @return bool
     */
    public function addRequest($paymentId, $type = null,
        $responseCode = null, $responseMessage = null,
        $responseJson = null)
    {

        if (isset($paymentId)) {

            $insertArray = array(
                'request_payment_id' => $paymentId,
            );

            if (isset($type)) {
                $insertArray['request_type'] = $type;
            }

            if (isset($responseCode)) {
                $insertArray['request_response_code'] = $responseCode;
            }

            if (isset($responseMessage)) {
                $insertArray['request_response_message'] = $responseMessage;
            }

            if (isset($responseJson)) {
                $insertArray['request_response_json'] = $responseJson;
            }

            $insertArray['request_added'] = date('Y-m-d H:i:s');

            $this->db->insert(self::TABLE_REQUESTS, $insertArray);

            return true;

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;
    }

    /**
     * @param int|null [$status = null]
     * @param string|null [$type = null]
     * @param string|null [$amount = null]
     * @param string|null [$token = null]
     * @param string|null [$paymentId = null]
     * @param int|null [$saleId = null]
     * @return int|bool
     */
    public function addPayment($status = null,
        $type = null, $amount = null, $token = null,
        $paymentId = null, $saleId = null)
    {

        $insertArray = array();

        if (isset($status)) {
            $insertArray['request_payment_status'] =
                $status;
        }

        if (isset($type)) {
            $insertArray['request_payment_type'] = $type;
        }

        if (isset($amount)) {
            $insertArray['request_payment_amount'] = $amount;
        }

        if (isset($token)) {
            $insertArray['request_payment_token'] = $token;
        }

        if (isset($paymentId)) {
            $insertArray['request_payment_paymentid'] = $paymentId;
        }

        if (isset($saleId)) {
            $insertArray['request_payment_sale_id'] = $saleId;
        }

        if (sizeof($insertArray) > 0) {

            $insertArray['request_payment_added'] = date('Y-m-d H:i:s');

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;

            $db->insert(self::TABLE_PAYMENTS, $insertArray);

            return $db->insert_id();

        }

        return false;

    }

    /**
     * @param int $id
     * @param int|null [$status = null]
     * @param int|null [$type = null]
     * @param string|null [$amount = null]
     * @param string|null [$token = null]
     * @param string|null [$paymentId = null]
     * @param int|null [$saleId = null]
     * @return bool
     */
    public function updatePayment($id, $status = null,
          $type = null, $amount = null, $token = null,
          $paymentId = null, $saleId = null)
    {
        if (isset($id)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;

            $updateArray = array();

            if (isset($status)) {
                $updateArray['request_payment_status'] = $status;
            }

            if (isset($type)) {
                $updateArray['request_payment_type'] = $type;
            }

            if (isset($amount)) {
                $updateArray['request_payment_amount'] = $amount;
            }

            if (isset($token)) {
                $updateArray['request_payment_token'] = $token;
            }

            if (isset($paymentId)) {
                $updateArray['request_payment_paymentid'] = $paymentId;
            }

            if (isset($saleId)) {
                $updateArray['request_payment_sale_id'] = $saleId;
            }

            if (sizeof($updateArray) > 0) {
                $db->update(self::TABLE_PAYMENTS, $updateArray, 'request_payment_id = ' . $db->escape_str($id), 1);
            }

            return true;


        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;
    }

    /**
     * @param int $requestId
     * @return bool
     */
    public function deleteRequest($requestId)
    {
        if (isset($requestId)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;

            $db->delete(self::TABLE_REQUESTS, 'request_id = ' . $db->escape_str($requestId), 1);

            return true;

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;
    }

    /**
     * @param int $paymentId
     * @return bool
     */
    public function deletePayment($paymentId)
    {
        if (isset($paymentId)) {

            /** @var \CI_DB_active_record $db */
            $db = & $this->db;

            $db->delete(self::TABLE_PAYMENTS, 'request_payment_id = ' . $db->escape_str($paymentId, 1));
            $db->delete(self::TABLE_REQUESTS, 'request_payment_id = ' . $db->escape_str($paymentId));

            return true;

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return false;
    }

    /**
     * @param string $type
     * @param string $totalAmount
     * @throws \Exception
     * @return \InitializePaymentRequest
     */
    public function initializePayment($type, $totalAmount)
    {

        if ($payment = $this->getUnverifiedPaymentByTypeAndAmount(
            $type,
            $totalAmount)
        ) {
            $invoiceNumber = $payment['request_payment_id'];
        } else {
            $invoiceNumber = $this->addPayment(
                null,
                $type,
                $totalAmount);
        }

        $settings = $this->getSettings();

        $values = array(
            'MerchantID' => $settings['merchant_id'],
            'Password' => $settings['password'],
            'TranType' => \InitializePaymentRequest::TRAN_TYPE_SALE,
            'TotalAmount' => $totalAmount,
            'Frequency' => \InitializePaymentRequest::FREQUENCY_ONE_TIME,
            'OperatorID' => $settings['operator_id'],
            'Invoice' => $invoiceNumber,
            'Memo' => self::MEMO_TAG,
            'TaxAmount' => (double) 0,
            'ProcessCompleteUrl' => self::getReturnUrl(),
            'ReturnUrl' => self::getReturnUrl(),
            'DefaultSwipe' => (!empty($settings['swipe']) ?
                \InitializePaymentRequest::DEFAULT_SWIPE_SWIPE :
                \InitializePaymentRequest::DEFAULT_SWIPE_MANUAL),
        );

        $request = new \InitializePaymentRequest($values);

        if ($request->validate()) {
            $request->process();
            return $request;
        } else {
            $specification =
                & $request->getSpecification();
            $values =
                $request->getValues();
            Throw new \Exception(
                'Request did not validate "' . print_r($specification->getValidationErrors(), true)
                . '", values: "' . print_r($values, true) . '" in ' . __METHOD__ . ' in ' . __FILE__);
        }

    }

    /**
     * @param int $invoice
     * @param string|null [$paymentId = null]
     * @param string|null [$returnCode = null]
     * @param string|null [$returnMessage = null]
     * @throws \Exception
     * @return \HostedCheckoutRequest|bool
     */
    public function hostedCheckout($invoice, $paymentId = null,
        $returnCode = null, $returnMessage = null)
    {

        if (isset($invoice)) {

            $request = new \HostedCheckoutRequest($invoice, $paymentId,
                $returnCode, $returnMessage);

            $request->process();
            $request->request();
            return $request;

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }

        return false;

    }

    /**
     * @param int $responseCode
     * @return string
     */
    public static function getResponseErrorMessage($responseCode)
    {
        if (isset($responseCode)) {
            $errorMessages = \MercuryRequest::getErrorMessages();
            return (isset($errorMessages[$responseCode]) ? $errorMessages[$responseCode] : '');
        } else {
            return '';
        }
    }

    /**
     * @param string $paymentId
     * @throws \Exception
     * @return bool|\VerifyPaymentRequest
     */
    public function verifyPayment($paymentId)
    {

        if (!empty($paymentId)) {

            $settings = $this->getSettings();

            $values = array(
                'MerchantID' => $settings['merchant_id'],
                'Password' => $settings['password'],
                'PaymentID' => $paymentId,
            );

            $request = new \VerifyPaymentRequest($values);

            if ($request->validate()) {
                $request->process();
                return $request;
            } else {
                $specification =
                    & $request->getSpecification();
                $values =
                    $request->getValues();
                Throw new \Exception(
                    'Request did not validate "' . print_r($specification->getValidationErrors(), true)
                        . '", values: "' . print_r($values, true) . '" in ' . __METHOD__ . ' in ' . __FILE__);
            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }

        return false;

    }

    /**
     * @param string $authCode
     * @param string $invoice
     * @param string $purchaseAmount
     * @param string $refNo
     * @param string $token
     * @param string|null [$cardHolderName = null]
     * @param string|null [$terminalName = null]
     * @throws \Exception
     * @return bool|\CreditVoidSaleTokenRequest
     */
    public function creditVoidSaleToken($authCode, $invoice, $purchaseAmount,
        $refNo, $token, $cardHolderName = null, $terminalName = null)
    {

        if (isset($authCode, $invoice, $purchaseAmount, $refNo, $token)) {

            $settings = $this->getSettings();

            $values = array(
                'AuthCode' => $authCode,
                'Frequency' => 'OneTime',
                'Invoice' => $invoice,
                'Memo' => self::MEMO_TAG,
                'MerchantID' => $settings['merchant_id'],
                'OperatorID' => $settings['operator_id'],
                'PurchaseAmount' => $purchaseAmount,
                'RefNo' => $refNo,
                'Token' => $token,
            );

            if (isset($cardHolderName)) {
                $values['CardHolderName'] = $cardHolderName;
            }

            if (isset($terminalName)) {
                $values['TerminalName'] = $terminalName;
            }

            $request = new \CreditVoidSaleTokenRequest(
                $values, $settings['password']);

            if ($request->validate()) {
                $request->process();
                return $request;
            } else {
                $specification =
                    & $request->getSpecification();
                $values =
                    $request->getValues();
                Throw new \Exception(
                    'Request did not validate "' . print_r($specification->getValidationErrors(), true)
                    . '", values: "' . print_r($values, true) . '" in ' . __METHOD__ . ' in ' . __FILE__);
            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }

        return false;

    }

    /**
     * @param int $saleId
     * @param string $purchaseAmount
     * @param string $token
     * @param string|null [$cardHolderName = null]
     * @param string|null [$terminalName = null]
     * @throws \Exception
     * @return bool|\CreditReturnTokenRequest
     */
    public function creditReturnToken($saleId, $purchaseAmount, $token,
        $cardHolderName = null, $terminalName = null)
    {

        if (isset($saleId, $purchaseAmount, $token)) {

            $settings = $this->getSettings();

            if ($paymentId = $this->addPayment(
                null,
                'Return',
                $purchaseAmount,
                $token,
                null,
                $saleId)
            ) {

                $values = array(
                    'MerchantID' => $settings['merchant_id'],
                    'Invoice' => $paymentId,
                    'Frequency' => 'OneTime',
                    'OperatorID' => $settings['operator_id'],
                    'Memo' => self::MEMO_TAG,
                    'PurchaseAmount' => $purchaseAmount,
                    'Token' => $token,
                );

                if (isset($cardHolderName)) {
                    $values['CardHolderName'] = $cardHolderName;
                }

                if (isset($terminalName)) {
                    $values['TerminalName'] = $terminalName;
                }

                $request = new \CreditReturnTokenRequest(
                    $values, $settings['password']);

                if ($request->validate()) {
                    $request->process();
                    return $request;
                } else {
                    $specification =
                        & $request->getSpecification();
                    $values =
                        $request->getValues();
                    Throw new \Exception(
                        'Request did not validate "' . print_r($specification->getValidationErrors(), true)
                        . '", values: "' . print_r($values, true) . '" in ' . __METHOD__ . ' in ' . __FILE__);
                }

            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }

        return false;

    }

    /**
     * @param string $authCode
     * @param int $saleId
     * @param float $purchaseAmount
     * @param string $refNo
     * @param string $token
     * @param string|null [$cardHolderName = null]
     * @param string|null [$terminalName = null]
     * @throws \Exception
     * @return bool|\CreditVoidReturnTokenRequest
     */
    public function creditVoidReturnToken($authCode, $saleId,
        $purchaseAmount, $refNo, $token,
        $cardHolderName = null, $terminalName = null)
    {

        if (isset($authCode, $saleId,
            $purchaseAmount, $refNo, $token)
        ) {

            $settings = $this->getSettings();

            if ($paymentId = $this->addPayment(
                null,
                'Void Return',
                $purchaseAmount,
                $token,
                null,
                $saleId)
            ) {

                $values = array(
                    'AuthCode' => $authCode,
                    'Frequency' => 'OneTime',
                    'Invoice' => $paymentId,
                    'Memo' => self::MEMO_TAG,
                    'MerchantID' => $settings['merchant_id'],
                    'OperatorID' => $settings['operator_id'],
                    'PurchaseAmount' => $purchaseAmount,
                    'RefNo' => $refNo,
                    'Token' => $token,
                );

                if (isset($cardHolderName)) {
                    $values['CardHolderName'] = $cardHolderName;
                }

                if (isset($terminalName)) {
                    $values['TerminalName'] = $terminalName;
                }

                $request = new \CreditVoidReturnTokenRequest(
                    $values, $settings['password']);

                if ($request->validate()) {
                    $request->process();
                    return $request;
                } else {
                    $specification =
                        & $request->getSpecification();
                    $values =
                        $request->getValues();
                    Throw new \Exception(
                        'Request did not validate "' . print_r($specification->getValidationErrors(), true)
                        . '", values: "' . print_r($values, true) . '" in ' . __METHOD__ . ' in ' . __FILE__);
                }

            }

        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }

        return false;

    }

    /**
     * @static
     * @return string
     */
    public static function getReturnUrl()
    {
        if (!empty($_SERVER['SERVER_NAME'])) {
            $server = $_SERVER['SERVER_NAME'];
        } else {
            $server = 'pos.cvj.se';
        }
        return base_url('index.php/PaymentGateway/ProcessResponse');
    }

    /**
     * @param \MercuryRequest $request
     * @throws \Exception
     * @return string               HTML markup
     */
    public function displayHostedCheckoutRedirect($request)
    {
        if (isset($request)) {
            $response = $request->getResponse();
            if (isset($response->PaymentID)) {
                return $this->getHostedCheckoutRedirectHTML($response->PaymentID);
            } else {
                Throw new \Exception(
                    'No PaymentID specified in request in ' . __METHOD__
                    . ' in ' . __FILE__);
            }
        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return '';
    }

    /**
     * @param string $paymentId
     * @throws \Exception
     * @return string               HTML markup
     */
    public function getHostedCheckoutRedirectHtml($paymentId)
    {
        if (isset($paymentId)) {
            if (self::isTesting()) {
                $url = 'https://hc.mercurydev.net/CheckoutPOS.aspx';
            } else {
                $url = 'https://hc.mercurypay.com/CheckoutPOS.aspx';
            }

            $https = (!empty($_SERVER['HTTPS']));

            return '<form name="frmCheckout" method="post" action="' . $url . '">'
            . '<div><input name="PaymentID" type="hidden" value="' . $paymentId . '" /></div>'
            . '<div><input name="ReturnMethod" type="hidden" value="' . (!empty($https) ? 'POST' : 'GET') . '" /></div>'
            . '</form>'
            . '<script type="text/javascript">$(document).ready(function() { document.frmCheckout.submit(); });</script>';
        } else {
            $this->_throwInvalidParametersException(__METHOD__);
        }
        return '';
    }

    /**
     * @param \MercuryRequest $request
     * @throws \Exception
     * @return string               HTML markup
     */
    public function displayHostedCheckoutIframe($request)
    {
        $response = $request->getResponse();
        if (isset($response->PaymentID)) {

            if (self::isTesting()) {
                $url = 'https://hc.mercurydev.net/CheckoutPOSiFrame.aspx';
            } else {
                $url = 'https://hc.mercurypay.com/CheckoutPOSiFrame.aspx';
            }
            $url .= '?pid=' . $response->PaymentID . '&ReturnMethod=get';

            $values = $request->getValues();

            if (isset($values['KeyPad'])
                && $values['KeyPad'] == 'On'
            ) {
                $width = '820px';
            } else {
                $width = '574px';
            }

            if (isset($values['AVS'])
                && $values['AVS'] == 'Off'
            ) {
                $height = '400px';
            } else if ((isset($values['AVSAddress'])
                && $values['AVSAddress'] == 'On')
                XOR (isset($values['Zip'])
                && $values['Zip'] == 'On')
            ) {
                $height = '460px';
            } else if (isset($values['AVSAddress'])
                && $values['AVSAddress'] == 'On'
                && isset($values['Zip'])
                && $values['Zip'] == 'On'
            ) {
                $height = '520px';
            } else {
                $height = '520px';
            }

            return '<div style="width: ' . $width . '">'
                . '<iframe id="ifrm" src="' . $url . '" width="' . $width . '" height="' . $height . '" '
                . 'scrolling="auto" frameborder="0" runat="server" style="text-align: right; display: none;" '
                . 'onload="this.style.display=\'block\';">'
                . 'Your browser does not support iFrames.  To view this content, please download and '
                . 'use the latest version of one of the following browsers:  Internet Explorer, '
                . 'Firefox, Google Chrome or Safari.'
                . '</iframe></div>';

        } else {
            Throw new \Exception(
                'No PaymentID specified in request in ' . __METHOD__
                    . ' in ' . __FILE__);
        }
    }

    /**
     * @return bool
     */
    private function _isInstalled()
    {

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;
        $db->from('modules');
        $db->where('module_id', self::MODULE_ID);
        $query = $db->get();

        $accBool = true;

        // Does module exists if allowed modules?
        $accBool = ($accBool && $query->num_rows() === 1);

        $db->from('permissions');
        $db->where('module_id', self::MODULE_ID);
        $db->where('person_id', 1);
        $query = $db->get();

        // Does module exist sin permitted modules for person 1?
        $accBool = ($accBool && $query->num_rows() === 1);

        // Does settings table exists?
        $accBool = ($accBool && $db->table_exists(self::TABLE_SETTINGS));

        // Does requests table exists?
        $accBool = ($accBool && $db->table_exists(self::TABLE_REQUESTS));

        // Does payments table exists?
        $accBool = ($accBool && $db->table_exists(self::TABLE_PAYMENTS));

        return $accBool;

    }

    /**
     * @return bool
     */
    private function _install()
    {

        $accBool = true;

        /** @var \CI_DB_active_record $db */
        $db = & $this->db;

        $db->from('modules');
        $db->where('module_id', self::MODULE_ID);
        $query = $db->get();

        if ($query->num_rows() !== 1) {

            if (!$db->insert('modules',
                array(
                    'name_lang_key' => 'module_payment_gateway',
                    'desc_lang_key' => 'module_payment_gateway_desc',
                    'sort' => '95',
                    'module_id' => self::MODULE_ID,
                ))
            ) {
                $accBool = false;
            }

        }

        $db->from('permissions');
        $db->where('module_id', self::MODULE_ID);
        $db->where('person_id', 1);
        $query = $db->get();

        if ($query->num_rows() !== 1) {

            if (!$db->insert('permissions',
                array(
                    'module_id' => self::MODULE_ID,
                    'person_id' => 1,
                ))
            ) {
                $accBool = false;
            }

        }

        if (!$db->table_exists(self::TABLE_SETTINGS)) {

            /** @var \CI_DB_forge $dbforge */
            $dbforge = $this->dbforge;

            $dbforge->add_field('setting_key VARCHAR(100) NULL DEFAULT NULL');
            $dbforge->add_field('setting_value LONGTEXT NULL DEFAULT NULL');

            if (!$dbforge->create_table(self::TABLE_SETTINGS, true)) {
                $accBool = false;
            }

        }


        if (!$db->table_exists(self::TABLE_PAYMENTS)) {

            /** @var \CI_DB_forge $dbforge */
            $dbforge = $this->dbforge;

            $dbforge->add_field('request_payment_id INT(11) UNSIGNED NOT NULL AUTO_INCREMENT');
            $dbforge->add_field('request_payment_paymentid VARCHAR(100) NULL DEFAULT NULL');
            $dbforge->add_field('request_payment_status INT(1) UNSIGNED NULL DEFAULT NULL');
            $dbforge->add_field('request_payment_type VARCHAR(100) NULL DEFAULT NULL');
            $dbforge->add_field('request_payment_amount VARCHAR(100) NULL DEFAULT NULL');
            $dbforge->add_field('request_payment_token LONGTEXT NULL DEFAULT NULL');
            $dbforge->add_field('request_payment_sale_id INT(11) UNSIGNED NULL DEFAULT NULL');
            $dbforge->add_field('request_payment_added DATETIME NULL DEFAULT NULL');
            $dbforge->add_field('request_payment_updated TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP');
            $dbforge->add_key('request_payment_id', true);

            if (!$dbforge->create_table(self::TABLE_PAYMENTS, true)) {
                $accBool = false;
            }

        }

        if (!$db->table_exists(self::TABLE_REQUESTS)) {

            /** @var \CI_DB_forge $dbforge */
            $dbforge = $this->dbforge;

            $dbforge->add_field('request_id INT(11) UNSIGNED NOT NULL AUTO_INCREMENT');
            $dbforge->add_field('request_payment_id INT(11) UNSIGNED NULL DEFAULT NULL');
            $dbforge->add_field('request_type INT(1) UNSIGNED NULL DEFAULT NULL');
            $dbforge->add_field('request_response_code VARCHAR(100) NULL DEFAULT NULL');
            $dbforge->add_field('request_response_message LONGTEXT NULL DEFAULT NULL');
            $dbforge->add_field('request_response_json LONGBLOB NULL DEFAULT NULL');
            $dbforge->add_field('request_added DATETIME NULL DEFAULT NULL');
            $dbforge->add_field('request_updated TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP');
            $dbforge->add_key('request_id', true);

            if (!$dbforge->create_table(self::TABLE_REQUESTS, true)) {
                $accBool = false;
            }

        }

        return $accBool;

    }

    /**
     * @param string|null [$method = null]
     * @throws Exception
     */
    private function _throwInvalidParametersException($method = null)
    {
        Throw new \Exception(
            'Invalid parameters ' . (!empty($method) ? 'for ' . $method : '')
            . ' in ' . __FILE__);
    }

}

