<?php
/**
 * @package Mercury Payment Integration
 */

require_once('secure_area.php');

/**
 * @var \Mercury $Mercury
 * @property \CI_DB_active_record $db
 */
class PaymentGateway extends Secure_area
{

    /**
     * @var string
     */
    const MODULE_ID = 'PaymentGateway';

    /**
     * @var int
     */
    const DEFAULT_PAYMENTS_PER_PAGE = 20;

    /**
     * @var string
     */
    const TABLE_SETTINGS = 'paymentgateway_settings';

    /**
     *
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     *
     */
    public function index()
    {
        $this->load->view('payment_gateway/index');
    }

    /**
     * @param int [$page = 1]
     */
    public function payments($page = 1)
    {

        /** @var Mercury $mercuryModel */
        $mercuryModel = $this->Mercury;
        $settings = $mercuryModel->getSettings();

        $data = array();

        $resultsPerPage =
            (!empty($settings['paymentsperpage']) ? $settings['paymentsperpage'] : self::DEFAULT_PAYMENTS_PER_PAGE);

        $searchFilter = '';
        if (isset($_GET['search_filter'])) {
            $searchFilter = strip_tags($_GET['search_filter']);
        }

        $totalCount = $mercuryModel->getPaymentsCount(
            $searchFilter);
        $totalPages = ceil($totalCount / $resultsPerPage);

        if (!is_numeric($page)
            || $page < 1
            || $page > $totalPages
        ) {
            $page = 1;
        }

        $payments = $mercuryModel->getPayments(
            ($page - 1) * $resultsPerPage, $resultsPerPage, $searchFilter);

        $count = (is_array($payments) ? sizeof($payments) : 0);

        $data['count'] = $count;
        $data['payments'] = $payments;
        $data['totalCount'] = $totalCount;
        $data['statuses'] = $mercuryModel->getStatuses();
        $data['page'] = $page;
        $data['totalPages'] = $totalPages;
        $data['searchFilter'] = $searchFilter;

        $this->load->view('payment_gateway/payments', $data);

    }

    /**
     *
     */
    public function settings()
    {

        /** @var Mercury $mercuryModel */
        $mercuryModel = $this->Mercury;

        // Do we have a form submit?
        if ($_SERVER['REQUEST_METHOD'] == 'POST'
            && isset($_POST)
            && is_array($_POST)
            && sizeof($_POST) > 0
        ) {

            // Gather submit
            $submit = array(
                'merchant_id' => (isset($_POST['paymentgateway_settings']['merchant_id']) ? strip_tags($_POST['paymentgateway_settings']['merchant_id']) : ''),
                'operator_id' => (isset($_POST['paymentgateway_settings']['operator_id']) ? strip_tags($_POST['paymentgateway_settings']['operator_id']) : ''),
                'password' => (isset($_POST['paymentgateway_settings']['password']) ? strip_tags($_POST['paymentgateway_settings']['password']) : ''),
                'testing_mode' => (!empty($_POST['paymentgateway_settings']['testing_mode']) ? true : false),
                'swipe' => (!empty($_POST['paymentgateway_settings']['swipe']) ? true : false),
                'paymentsperpage' => (!empty($_POST['paymentgateway_settings']['paymentsperpage']) && is_numeric($_POST['paymentgateway_settings']['paymentsperpage']) ? (int) $_POST['paymentgateway_settings']['paymentsperpage'] : self::DEFAULT_PAYMENTS_PER_PAGE),
            );

            $mercuryModel->setSettings($submit);

        }

        $settings = $mercuryModel->getSettings();

        $this->load->view('payment_gateway/settings', array('settings' => $settings));

    }

    /**
     * @throws \Exception
     */
    public function makeTransaction()
    {
        if (isset($_GET['id'])
            && isset($_GET['amount'])
        ) {

            /** @var Mercury $mercuryModel */
            $mercuryModel = $this->Mercury;

            $type = $_GET['id'];
            $totalAmount = $mercuryModel->formatAmount($_GET['amount']);

            if ($mercuryModel->availablePaymentExistsByTypeAndAmount($type, $totalAmount)) {
                $this->_display('Verified payment already exists by type and amount.');
            } else {

                $request = $mercuryModel->initializePayment($type, $totalAmount);
                $request->request();

                if ($request->responseWasSuccessful()) {

                    $html = $mercuryModel->displayHostedCheckoutRedirect($request);
                    $this->load->view('payment_gateway/hosted_checkout', array('html' => $html));

                } else {
                    $this->_display(
                    'Failed with response. Response: "' . print_r($request, true) . '"'
                    . ', model: "' . print_r($mercuryModel, true) . '"');
                }

            }

        } else {
            Throw new \Exception(
                'Invalid parameters in ' . __METHOD__
                . ' in ' . __FILE__);
        }
    }

    /**
     * @param int $paymentId
     * @throws \Exception
     */
    public function deletePayment($paymentId)
    {

        if (isset($paymentId)) {

            /** @var Mercury $mercuryModel */
            $mercuryModel = $this->Mercury;

            if ($payment = $mercuryModel->getPaymentById($paymentId)) {

                if ($mercuryModel->deletePayment($paymentId)) {
                    $this->_display('Successfully deleted payment and associated requests.');
                } else {
                    Throw new \Exception(
                        'Failed to delete payment with id ' . $paymentId
                        . ' in ' . __FILE__);
                }
            } else {
                $this->_display(
                    'Could not find any matching payment for request.');
            }

        } else {
            Throw new \Exception(
                'No payment Id specified in request');
        }

    }

    /**
     * @param int $paymentId
     * @throws \Exception
     */
    public function returnSale($paymentId)
    {

        if (isset($paymentId)) {

            /** @var Mercury $mercuryModel */
            $mercuryModel = $this->Mercury;

            if ($payment = $mercuryModel->getVerifiedPaymentById($paymentId)) {

                $saleId = (int) $payment['request_payment_sale_id'];

                if ($verifiedRequest = $mercuryModel->getVerifiedRequestById($paymentId)) {

                    if ($requestJson = json_decode($verifiedRequest['request_response_json'], true))
                    {
                        if (!empty($requestJson['Invoice'])
                            && !empty($requestJson['AuthAmount'])
                            && !empty($requestJson['Token'])
                        ) {
                            if ($request = $mercuryModel->creditReturnToken(
                                $saleId,
                                (double) $requestJson['AuthAmount'],
                                (string) $requestJson['Token'])
                            ) {

                                $request->request();
                                $wasSuccessFul = $request->responseWasSuccessful();

                                if ($wasSuccessFul) {
                                    $this->_display('Return sale request was successful.');
                                } else {
                                    $this->_display('Return sale request failed.');
                                }

                            } else {
                                $this->_display(
                                    'Failed to make a return sale request. Try again.');
                            }
                        } else {
                            $this->_display(
                                'Request data lacks required fields for return sale request.');
                        }
                    } else {
                        Throw new \Exception(
                            'Failed to decode data for verified request');
                    }
                } else {
                    Throw new \Exception(
                        'Could not find any matching verified request');
                }
            } else {
                $this->_display(
                    'Could not find any matching verified payment for request.');
            }

        } else {
            Throw new \Exception(
                'No payment Id specified in request');
        }

    }

    /**
     * @param int $paymentId
     * @throws \Exception
     */
    public function voidSale($paymentId)
    {

        if (isset($paymentId)) {

            /** @var Mercury $mercuryModel */
            $mercuryModel = $this->Mercury;

            if ($payment = $mercuryModel->getVerifiedPaymentById($paymentId)) {

                if ($verifiedRequest = $mercuryModel->getVerifiedRequestById($paymentId)) {

                    if ($requestJson = json_decode($verifiedRequest['request_response_json'], true))
                    {
                        if (!empty($requestJson['AuthCode'])
                            && !empty($requestJson['Invoice'])
                            && !empty($requestJson['AuthAmount'])
                            && !empty($requestJson['RefNo'])
                            && !empty($requestJson['Token'])
                        ) {
                            if ($request = $mercuryModel->creditVoidSaleToken(
                                (string) $requestJson['AuthCode'],
                                (string) $requestJson['Invoice'],
                                (double) $requestJson['AuthAmount'],
                                (string) $requestJson['RefNo'],
                                (string) $requestJson['Token'])
                            ) {

                                $request->request();
                                $wasSuccessFul = $request->responseWasSuccessful();

                                if ($wasSuccessFul) {
                                    $this->_display('Void sale request was successful.');
                                } else {
                                    $this->_display('Void sale request failed.');
                                }

                            } else {
                                $this->_display(
                                    'Failed to make a void sale request. Try again.');
                            }
                        } else {
                            $this->_display(
                                'Request data lacks required fields for void sale request.');
                        }
                    } else {
                        Throw new \Exception(
                            'Failed to decode data for verified request');
                    }
                } else {
                    Throw new \Exception(
                        'Could not find any matching verified request');
                }
            } else {
                $this->_display(
                    'Could not find any matching verified payment for request.');
            }

        } else {
            Throw new \Exception(
                'No payment Id specified in request');
        }
    }

    /**
     * @param int $paymentId
     * @throws \Exception
     */
    public function voidReturn($paymentId)
    {

        if (isset($paymentId)) {

            /** @var Mercury $mercuryModel */
            $mercuryModel = $this->Mercury;

            if ($payment = $mercuryModel->getReturnedPaymentById($paymentId)) {

                $saleId = (int) $payment['request_payment_sale_id'];

                if ($returnedRequest = $mercuryModel->getReturnedRequestById(
                    $paymentId)
                ) {

                    if ($requestJson = json_decode(
                        $returnedRequest['request_response_json'], true)
                    ) {
                        if (!empty($requestJson['AuthCode'])
                            && !empty($requestJson['Invoice'])
                            && !empty($requestJson['PurchaseAmount'])
                            && !empty($requestJson['RefNo'])
                            && !empty($requestJson['Token'])
                        ) {

                            if ($request = $mercuryModel->creditVoidReturnToken(
                                (string) $requestJson['AuthCode'],
                                $saleId,
                                (double) $requestJson['PurchaseAmount'],
                                (string) $requestJson['RefNo'],
                                (string) $requestJson['Token'])
                            ) {

                                $request->request();
                                $wasSuccessFul = $request->responseWasSuccessful();

                                if ($wasSuccessFul) {
                                    $this->_display('Void return request was successful.');
                                } else {
                                    $this->_display('Void return request failed.');
                                }

                            } else {
                                $this->_display(
                                    'Failed to make a void return request. Try again.');
                            }

                        } else {
                            $this->_display(
                                'Request data lacks required fields for void return request.');
                        }
                    } else {
                        Throw new \Exception(
                            'Failed to decode data for verified request');
                    }
                } else {
                    Throw new \Exception(
                        'Could not find any matching verified request');
                }
            } else {
                $this->_display(
                    'Could not find any matching returned payment for request.');
            }
        } else {
            Throw new \Exception(
                'No payment Id specified in request');
        }

    }

    /**
     *
     */
    public function processResponse()
    {

        $requestMethodIsPost = (isset($_SERVER['REQUEST_METHOD'])
            && $_SERVER['REQUEST_METHOD'] == 'POST');

        if ($requestMethodIsPost) {
            $paymentId = (isset($_POST['PaymentID']) ? $_POST['PaymentID'] : '');
            $returnCode = (isset($_POST['ReturnCode']) ? (int) $_POST['ReturnCode'] : null);
            $returnMessage = (isset($_POST['ReturnMessage']) ? $_POST['ReturnMessage'] : '');
        } else {
            $paymentId = (isset($_GET['PaymentID']) ? $_GET['PaymentID'] : '');
            $returnCode = (isset($_GET['ReturnCode']) ? (int) $_GET['ReturnCode'] : null);
            $returnMessage = (isset($_GET['ReturnMessage']) ? $_GET['ReturnMessage'] : '');
        }

        if (!empty($paymentId)
            && isset($returnCode)
            && (!$requestMethodIsPost
            || !empty($returnMessage))
        ) {

            /** @var Mercury $mercuryModel */
            $mercuryModel = $this->Mercury;

            if ($payment = $mercuryModel->getPaymentByPaymentId(
                $paymentId)
            ) {

                if ($payment['request_payment_status'] ==
                    $mercuryModel::REQUEST_STATUS_INITIATED_SUCCESS
                ) {

                    $invoice = (int) $payment['request_payment_id'];

                    if ($request = $mercuryModel->hostedCheckout(
                        $invoice,
                        $paymentId,
                        $returnCode,
                        $returnMessage)
                    ) {

                        $wasSuccessful = $request->responseWasSuccessful();

                        if ($wasSuccessful) {

                            $this->_display('Hosted checkout request was successful.');

                            $request = $mercuryModel->verifyPayment($paymentId);
                            $request->request();

                            $wasSuccessful = $request->responseWasSuccessful();

                            if ($wasSuccessful) {
                                $this->_display('Payment verification was successful.');
                            } else {
                                $this->_display('Payment verification failed.');
                            }

                            $this->_display('Please close this window.');

                        } else {


                            $response = $request->getResponse();
                            $returnCode = (isset($response->ReturnCode) ? $response->ReturnCode : null);
                            $returnMessage = (isset($returnCode) ? $mercuryModel::getResponseErrorMessage($returnCode) : '');

                            $this->_display('Hosted checkout request failed with return code '
                            . $returnCode . ' with message "' . $returnMessage . '"');

                        }
                    } else {
                        Throw new \Exception(
                            'Failed to build response object in ' . __METHOD__
                            . ' in ' . __FILE__);
                    }

                } else {
                    $this->_display(
                        'This request has already been processed.');
                }
            } else {
                Throw new \Exception(
                    'Could not find initiated payment by PaymentID: "' . $paymentId . '" in '
                    . __METHOD__ . ' in ' . __FILE__);
            }
        } else {
            Throw new \Exception(
                'Invalid response in POST: "' . print_r($_POST, true) . '" OR GET: "' . print_r($_GET, true) . '" in '
                . __METHOD__ . ' in ' . __FILE__);
        }

    }

    /**
     * @param string $message
     */
    private function _display($message)
    {
        echo '<p>' . $message . '</p>' . "\n";
    }

}

